import tkinter as tk
from tkinter import messagebox, ttk
from bank_account import BankAccount

DB_FILE = "accounts.txt"

class MainWindow:
    def __init__(self, master):
        self.master = master
        self.master.title("Bank Account Manager")

        # Variables
        self.account_number_var = tk.StringVar()
        self.account_holder_var = tk.StringVar()
        self.account_type_var = tk.StringVar()
        self.balance_var = tk.StringVar()

        # Data
        self.accounts = BankAccount.load_accounts(DB_FILE)

        # UI Setup
        self.setup_title()
        self.setup_form()
        self.setup_buttons()
        self.setup_table()
        self.update_table()

    def setup_title(self):
        """Set up the window title label."""
        title = tk.Label(self.master, text="Bank Account Manager", font=("Arial", 20, "bold"))
        title.pack(pady=10)

    def setup_form(self):
        """Set up the form for account data entry."""
        form_frame = ttk.Frame(self.master, padding=10)
        form_frame.pack(fill="x")

        ttk.Label(form_frame, text="Account Number:").grid(row=0, column=0, sticky="e", padx=5, pady=5)
        self.account_number_entry = tk.Entry(form_frame, textvariable=self.account_number_var, width=30)
        self.account_number_entry.grid(row=0, column=1, sticky="w", padx=5, pady=5)

        ttk.Label(form_frame, text="Account Holder:").grid(row=1, column=0, sticky="e", padx=5, pady=5)
        self.account_holder_entry = tk.Entry(form_frame, textvariable=self.account_holder_var, width=30)
        self.account_holder_entry.grid(row=1, column=1, sticky="w", padx=5, pady=5)

        ttk.Label(form_frame, text="Account Type:").grid(row=2, column=0, sticky="e", padx=5, pady=5)
        self.account_type_entry = ttk.Combobox(form_frame, textvariable=self.account_type_var, values=["Saving", "Other"], state="readonly", width=28)
        self.account_type_entry.current(0)
        self.account_type_entry.grid(row=2, column=1, sticky="w", padx=5, pady=5)

        ttk.Label(form_frame, text="Balance:").grid(row=3, column=0, sticky="e", padx=5, pady=5)
        self.balance_entry = tk.Entry(form_frame, textvariable=self.balance_var, width=30)
        self.balance_entry.grid(row=3, column=1, sticky="w", padx=5, pady=5)

    def setup_buttons(self):
        """Set up Create, Update, Delete, and Withdraw/Deposit buttons."""
        btn_frame = ttk.Frame(self.master, padding=10)
        btn_frame.pack(fill="x")

        self.create_button = tk.Button(btn_frame, text="Create Account", command=self.create_account, width=15)
        self.update_button = tk.Button(btn_frame, text="Update Account", command=self.update_account, width=15)
        self.delete_button = tk.Button(btn_frame, text="Delete Account", command=self.delete_account, width=15)
        self.transact_button = tk.Button(btn_frame, text="Withdraw/Deposit", command=self.open_transact_window, width=15)

        self.create_button.grid(row=0, column=0, padx=5, pady=5)
        self.update_button.grid(row=0, column=1, padx=5, pady=5)
        self.delete_button.grid(row=0, column=2, padx=5, pady=5)
        self.transact_button.grid(row=0, column=3, padx=5, pady=5)
        self.update_button.grid_remove()
        self.delete_button.grid_remove()
        self.transact_button.grid_remove()

    def setup_table(self):
        """Set up the table to display accounts."""
        table_frame = ttk.Frame(self.master, padding=10)
        table_frame.pack(fill="both", expand=True)

        columns = ("AccountNumber", "AccountHolder", "AccountType", "Balance")
        self.tree = ttk.Treeview(table_frame, columns=columns, show="headings")
        for col in columns:
            self.tree.heading(col, text=col.replace("Account", "Account ").replace("Holder", "Holder").replace("Type", "Type").replace("Balance", "Balance"))
            self.tree.column(col, anchor="center", width=150)
        self.tree.pack(side="left", fill="both", expand=True)

        # Add vertical scrollbar
        scrollbar = ttk.Scrollbar(table_frame, orient="vertical", command=self.tree.yview)
        self.tree.configure(yscroll=scrollbar.set)
        scrollbar.pack(side="right", fill="y")

        self.tree.bind("<<TreeviewSelect>>", self.on_row_select)

    def create_account(self):
        """Create a new account and save it."""
        account_number = self.account_number_var.get()
        account_holder = self.account_holder_var.get()
        account_type = self.account_type_var.get()
        try:
            balance = float(self.balance_var.get())
            if not account_number or not account_holder:
                messagebox.showerror("Error", "Account number and holder are required.")
                return
            # Prevent duplicate account numbers
            if any(acc.account_number == account_number for acc in self.accounts):
                messagebox.showerror("Error", "Account number already exists.")
                return
            new_account = BankAccount(account_number, account_holder, account_type, balance)
            self.accounts.append(new_account)
            BankAccount.save_accounts(DB_FILE, self.accounts)
            self.update_table()
            messagebox.showinfo("Success", f"Account created for {account_holder}!")
            self.clear_entries()
        except ValueError:
            messagebox.showerror("Error", "Please enter a valid balance.")

    def update_account(self):
        """Update the selected account."""
        selected = self.tree.selection()
        if selected:
            idx = self.tree.index(selected[0])
            try:
                balance = float(self.balance_var.get())
                self.accounts[idx].account_holder = self.account_holder_var.get()
                self.accounts[idx].account_type = self.account_type_var.get()
                self.accounts[idx].balance = balance
                BankAccount.save_accounts(DB_FILE, self.accounts)
                self.update_table()
                messagebox.showinfo("Success", "Account updated!")
                self.clear_entries()
            except ValueError:
                messagebox.showerror("Error", "Please enter a valid balance.")

    def delete_account(self):
        """Delete the selected account."""
        selected = self.tree.selection()
        if selected:
            idx = self.tree.index(selected[0])
            del self.accounts[idx]
            BankAccount.save_accounts(DB_FILE, self.accounts)
            self.update_table()
            messagebox.showinfo("Success", "Account deleted!")
            self.clear_entries()

    def update_table(self):
        """Refresh the table with current account data."""
        for row in self.tree.get_children():
            self.tree.delete(row)
        for acc in self.accounts:
            self.tree.insert("", "end", values=(acc.account_number, acc.account_holder, acc.account_type, f"{acc.balance:.2f}"))

    def clear_entries(self):
        """Clear all entry fields and reset button states."""
        self.account_number_var.set("")
        self.account_holder_var.set("")
        self.account_type_var.set("Saving")
        self.balance_var.set("")
        self.account_number_entry.config(state="normal")
        self.update_button.grid_remove()
        self.delete_button.grid_remove()
        self.transact_button.grid_remove()
        self.create_button.grid(row=0, column=0, padx=5, pady=5)

    def on_row_select(self, event):
        """Fill entry fields with selected row and update button states."""
        selected = self.tree.selection()
        if selected:
            values = self.tree.item(selected[0], "values")
            self.account_number_var.set(values[0])
            self.account_holder_var.set(values[1])
            self.account_type_var.set(values[2])
            self.balance_var.set(values[3])
            self.account_number_entry.config(state="readonly")
            self.create_button.grid_remove()
            self.update_button.grid(row=0, column=1, padx=5, pady=5)
            self.delete_button.grid(row=0, column=2, padx=5, pady=5)
            self.transact_button.grid(row=0, column=3, padx=5, pady=5)
        else:
            self.clear_entries()

    def open_transact_window(self):
        """Open a new window for Withdraw/Deposit."""
        selected = self.tree.selection()
        if not selected:
            messagebox.showerror("Error", "Please select an account first.")
            return
        idx = self.tree.index(selected[0])
        acc = self.accounts[idx]

        # Create a new top-level window
        transact_win = tk.Toplevel(self.master)
        transact_win.title("Withdraw/Deposit")
        transact_win.geometry("300x250")
        transact_win.resizable(False, False)
        
        # Ensure this window stays on top and is modal
        transact_win.transient(self.master)
        transact_win.grab_set()

        # Main frame to organize widgets
        main_frame = ttk.Frame(transact_win, padding=10)
        main_frame.pack(fill="both", expand=True)

        # Account information
        ttk.Label(main_frame, text=f"Account: {acc.account_number}", font=("Arial", 12)).pack(pady=5)
        ttk.Label(main_frame, text=f"Holder: {acc.account_holder}", font=("Arial", 12)).pack(pady=5)
        ttk.Label(main_frame, text=f"Current Balance: ${acc.balance:.2f}", font=("Arial", 12)).pack(pady=5)

        # Transaction type selection
        op_frame = ttk.Frame(main_frame)
        op_frame.pack(pady=5, fill="x")
        ttk.Label(op_frame, text="Transaction Type:").pack(side="left", padx=5)
        op_var = tk.StringVar(transact_win, value="Deposit")  # Explicitly set master
        ttk.Combobox(op_frame, textvariable=op_var, values=["Deposit", "Withdraw"], state="readonly", width=10).pack(side="left", padx=5)

        # Amount entry
        amount_frame = ttk.Frame(main_frame)
        amount_frame.pack(pady=5, fill="x")
        ttk.Label(amount_frame, text="Amount:").pack(side="left", padx=5)
        amount_var = tk.StringVar(transact_win)  # Explicitly set master
        amount_entry = ttk.Entry(amount_frame, textvariable=amount_var, width=15)
        amount_entry.pack(side="left", padx=5)
        amount_entry.focus()

        def do_transact():
            try:
                amount_text = amount_var.get().strip()
                if not amount_text:
                    messagebox.showerror("Error", "Amount is required.")
                    return
                
                amt = float(amount_text)
                if amt <= 0:
                    messagebox.showerror("Error", "Amount must be positive.")
                    return
                
                if op_var.get() == "Deposit":
                    acc.balance += amt
                else:  # Withdraw
                    if amt > acc.balance:
                        messagebox.showerror("Error", "Insufficient funds.")
                        return
                    acc.balance -= amt
                
                BankAccount.save_accounts(DB_FILE, self.accounts)
                self.update_table()
                messagebox.showinfo("Success", f"{op_var.get()} of ${amt:.2f} successful!")
                transact_win.destroy()
                self.clear_entries()
            except ValueError:
                messagebox.showerror("Error", "Please enter a valid numeric amount.")
        
        # Submit button
        ttk.Button(main_frame, text="Submit", command=do_transact, width=15).pack(pady=15)

if __name__ == "__main__":
    root = tk.Tk()
    root.geometry("800x500")
    app = MainWindow(root)
    root.mainloop()
